<?php

namespace App\Http\Controllers\Concerns;

use App\Models\Transaction;
use App\Models\SalesDeletion;
use Illuminate\Support\Facades\DB;

trait DeletesSalesWithAudit
{
    protected function archiveAndSoftDeleteTransaction(Transaction $tx): void
    {
        DB::transaction(function () use ($tx) {
            // Build a full snapshot (transaction + lines + payments)
            $tx->loadMissing(['contact', 'location', 'lines', 'payments']);

            $snapshot = [
                'transaction' => $tx->toArray(),
                'lines'       => $tx->lines->toArray(),
                'payments'    => $tx->payments->toArray(),
            ];

            SalesDeletion::create([
                'transaction_id'       => $tx->id,
                'invoice_no'           => $tx->invoice_no ?? null,
                'contact_id'           => $tx->contact_id ?? null,
                'contact_name'         => optional($tx->contact)->name,
                'business_location_id' => $tx->location_id ?? null,
                'final_total'          => $tx->final_total ?? 0,
                'payment_status'       => $tx->payment_status ?? null,
                'deleted_by'           => auth()->id(),
                'deleted_at'           => now(),
                'payload'              => $snapshot,
            ]);

            // Soft-delete the transaction (keeps inventory rollbacks handled by your existing logic if it’s in observers)
            $tx->delete();
        });
    }

    // List trashed sales
    public function trashIndex()
    {
        $rows = \App\Models\Transaction::onlyTrashed()
            ->latest('deleted_at')
            ->paginate(25);

        return view('sells.trash', compact('rows'));
    }

    // Restore a trashed sale
    public function restore($id)
    {
        $tx = \App\Models\Transaction::onlyTrashed()->findOrFail($id);
        $tx->restore();

        // You can optionally reverse any inventory adjustments here if your app requires.

        return back()->with('status', 'Sale restored.');
    }

    // Permanently delete (and keep the audit row)
    public function forceDelete($id)
    {
        $tx = \App\Models\Transaction::onlyTrashed()->findOrFail($id);
        $tx->forceDelete();

        return back()->with('status', 'Sale permanently removed (audit kept).');
    }
}
